#version 130
#extension GL_EXT_gpu_shader4 : enable
// the version and open GL extension
// should be the first line of the shader
/////////////////////////////////////////////////////////////////////////////////
// Sunset Ocean - ReflectionsMod01.fsh    by   computerK   
//https://www.shadertoy.com/view/l3yBDd
//Licence : Creative Commons Attribution-ShareAlike 4.0
//http://creativecommons.org/licences/by-sa/4.0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D texture0;
uniform sampler2D texture1;
uniform sampler2D texture2;
uniform sampler2D texture3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

//------------------------------------------------------------------------------
// OCEAN SHADER - SUNSET WAVES WITH FOAM
//------------------------------------------------------------------------------
// Author: afl_ext 2017-2024
// Updated: computerK 2-3-2025
// License: MIT
// Description: Realistic ocean shader with sunset atmosphere, wave effects, and foam
// Features:
// - Dynamic wave generation with LOD
// - Realistic sunset atmosphere
// - Foam particles based on wave intensity
// - Raymarching water surface
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// CONSTANTS AND UNIFORMS
//------------------------------------------------------------------------------

// Wave Parameters
const float DRAG_MULT = 0.38; // Wave-induced displacement scaling
const float WATER_DEPTH = 1.0; // Vertical distance from surface to bottom
const int WAVE_DETAIL = 16; // Iterations for wave summing
const int WAVE_NORMAL_DETAIL = 36; // Iterations for normal calculations
const float WAVE_INTENSITY = 1.0; // Overall wave amplitude scaling

// Foam Parameters
const float FOAM_THRESHOLD_START = 0.7; // Wave value where foam begins
const float FOAM_THRESHOLD_END = 1.0; // Wave value where foam is full

// Camera Parameters
const float CAMERA_HEIGHT = 1.5; // Vertical camera position

// Shadertoy Built-in Uniforms
uniform float uMaxDepth; // Maximum depth for normalization

//------------------------------------------------------------------------------
// UTILITY FUNCTIONS
//------------------------------------------------------------------------------

// Calculates wave value and its derivative, based on the wave direction, position in space, wave frequency and time
vec2 calculateWave(vec2 position, vec2 direction, float frequency, float timeshift) {
  float x = dot(direction, position) * frequency + timeshift;
  float wave = exp(sin(x) - 1.0);
  float dx = wave * cos(x);
  return vec2(wave, -dx);
}

// Calculates waves by summing octaves of various waves with various parameters
float getwaves(vec2 position, int iterations) {
  float wavePhaseShift = length(position) * 0.1; // this is to avoid every octave having exactly the same phase everywhere
  float waveIteration = 0.0; // used for generating well distributed wave directions
  float frequency = 1.0; // frequency of the wave, this will change every iteration
  float timeMultiplier = 2.0; // time multiplier for the wave, this will change every iteration
  float weight = 1.0;// weight in final sum for the wave, this will change every iteration
  float sumOfValues = 0.0; // will store final sum of values
  float sumOfWeights = 0.0; // will store final sum of weights
  for(int i = 0; i < iterations; i++) {
    // generate a pseudo-random wave direction based on current iteration
    vec2 p = vec2(sin(waveIteration), cos(waveIteration));
    
    // calculate wave data using the refactored function
    vec2 res = calculateWave(position, p, frequency, iTime * timeMultiplier + wavePhaseShift);

    // shift position around according to wave drag and derivative of the wave
    position += p * res.y * weight * DRAG_MULT;

    // add the results to sums
    sumOfValues += res.x * weight;
    sumOfWeights += weight;

    // modify next octave ;
    weight = mix(weight, 0.0, 0.2);
    frequency *= 1.18;
    timeMultiplier *= 1.07;

    // increment waveIteration to vary the wave direction
    waveIteration += 1232.399963;
  }
  // calculate and return
  return sumOfValues / sumOfWeights;
}

// New function to reduce tiling by combining waves at different levels of detail
float getwavesLOD(vec2 position) {
    float total = 0.0;
    float weightTotal = 0.0;
    float dist = length(position);
    int lodLevels = 4; // default LOD levels
    if (dist < 30.0) {
        lodLevels = 7; // highest detail for very close proximity
    } else if (dist < 70.0) {
        lodLevels = 6; // high detail for close proximity
    } else if (dist < 150.0) {
        lodLevels = 5; // moderate detail for mid distance
    }
    for (int i = 0; i < lodLevels; i++) {
        float scale = pow(2.0, float(i));
        float weight = 1.0 / scale;
        total += getwaves(position * scale, WAVE_DETAIL) * weight;
        weightTotal += weight;
    }
    return total / weightTotal;
}

// Raymarches the ray from top water layer boundary to low water layer boundary
float raymarchwater(vec3 camera, vec3 start, vec3 end, float depth) {
  vec3 pos = start;
  vec3 dir = normalize(end - start);
  for(int i=0; i < 64; i++) {
    // the height is from 0 to -depth
    float height = getwavesLOD(pos.xz) * depth * WAVE_INTENSITY - depth;
    // if the waves height nearly matches the ray height, assume it's a hit and return the hit distance
    if(height + 0.01 > pos.y) {
      return distance(pos, camera);
    }
    // iterate forwards according to the height mismatch
    pos += dir * (pos.y - height);
  }
  // if hit was not registered, just assume hit the top layer, 
  // this makes the raymarching faster and looks better at higher distances
  return distance(start, camera);
}

// Calculate normal at point by calculating the height at the pos and 2 additional points very close to pos
vec3 normal(vec2 pos, float e, float depth) {
  vec2 ex = vec2(e, 0);
  float H = getwaves(pos.xy, WAVE_NORMAL_DETAIL) * depth;
  vec3 a = vec3(pos.x, H, pos.y);
  return normalize(
    cross(
      a - vec3(pos.x - e, getwaves(pos.xy - ex.xy, WAVE_NORMAL_DETAIL) * depth, pos.y), 
      a - vec3(pos.x, getwaves(pos.xy + ex.yx, WAVE_NORMAL_DETAIL) * depth, pos.y + e)
    )
  );
}

// Helper function generating a rotation matrix around the axis by the angle
mat3 createRotationMatrixAxisAngle(vec3 axis, float angle) {
  float s = sin(angle);
  float c = cos(angle);
  float oc = 1.0 - c;
  return mat3(
    oc * axis.x * axis.x + c, oc * axis.x * axis.y - axis.z * s, oc * axis.z * axis.x + axis.y * s, 
    oc * axis.x * axis.y + axis.z * s, oc * axis.y * axis.y + c, oc * axis.y * axis.z - axis.x * s, 
    oc * axis.z * axis.x - axis.y * s, oc * axis.y * axis.z + axis.x * s, oc * axis.z * axis.z + c
  );
}

// Helper function that generates camera ray based on UV and mouse
vec3 getRay(vec2 fragCoord) {
  vec2 uv = ((fragCoord.xy / iResolution.xy) * 2.0 - 1.0) * vec2(iResolution.x / iResolution.y, 1.0);
  // for fisheye, uncomment following line and comment the next one
  //vec3 proj = normalize(vec3(uv.x, uv.y, 1.0) + vec3(uv.x, uv.y, -1.0) * pow(length(uv), 2.0) * 0.05);  
  vec3 proj = normalize(vec3(uv.x, uv.y, 1.5));
  if(iResolution.x < 600.0) {
    return proj;
  }

  // Fixed rotation to face Sun direction (empirically determined)
  float fixedAngle = 250.0; // Optimal value for perfect alignment
  return createRotationMatrixAxisAngle(vec3(0.0, -1.0, 0.0), fixedAngle) 
    * proj;
}

// Ray-Plane intersection checker
float intersectPlane(vec3 origin, vec3 direction, vec3 point, vec3 normal) { 
  return clamp(dot(point - origin, normal) / dot(direction, normal), -1.0, 9991999.0); 
}

// Improved atmosphere function for sunset
vec3 computeSunsetAtmosphere(vec3 viewDir, vec3 sunDir) {
    // Normalize directions
    viewDir = normalize(viewDir);
    sunDir = normalize(sunDir);
    
    // Calculate the cosine of the angle between the view and sun directions
    float cosAngle = dot(viewDir, sunDir);
    
    // Use smoothstep to generate a smooth transition factor based on the angle
    float scattering = smoothstep(0.0, 1.0, 1.0 - clamp(cosAngle, 0.0, 1.0));
    
    // Define two characteristic colors for the sunset
    vec3 sunriseColor = vec3(1.0, 0.5, 0.0);   // vibrant orange
    vec3 twilightColor = vec3(0.8, 0.6, 0.4);    // softer, muted tone
    
    // Blend between the two colors based on the scattering factor
    vec3 atmosphereColor = mix(sunriseColor, twilightColor, scattering);
    
    // Apply an exponential falloff to simulate atmospheric density
    float density = exp(-scattering * 2.0);
    atmosphereColor *= density;
    
    return atmosphereColor;
}

// Some very barebones but fast atmosphere approximation
vec3 extra_cheap_atmosphere(vec3 raydir, vec3 sundir) {
  //sundir.y = max(sundir.y, -0.07);
  float special_trick = 1.0 / (raydir.y * 1.0 + 0.1);
  float special_trick2 = 1.0 / (sundir.y * 11.0 + 1.0);
  float raysundt = pow(abs(dot(sundir, raydir)), 2.0);
  float sundt = pow(max(0.0, dot(sundir, raydir)), 8.0);
  float mymie = sundt * special_trick * 0.2;
  vec3 suncolor = mix(vec3(1.0), max(vec3(0.0), vec3(1.0) - vec3(5.5, 13.0, 22.4) / 22.4), special_trick2);
  vec3 bluesky= vec3(5.5, 13.0, 22.4) / 22.4 * suncolor;
  vec3 bluesky2 = max(vec3(0.0), bluesky - vec3(5.5, 13.0, 22.4) * 0.002 * (special_trick + -6.0 * sundir.y * sundir.y));
  bluesky2 *= special_trick * (0.24 + raysundt * 0.24);
  return bluesky2 * (1.0 + 1.0 * pow(1.0 - raydir.y, 3.0));
} 

// Calculate where the sun should be, now fixed for a sunset effect
vec3 getSunDirection() {
    return normalize(vec3(-0.707, 0.25, 0.707)); // Lowered and fixed sun position
}

// Get atmosphere color for sunset
vec3 getAtmosphere(vec3 dir) {
    vec3 sundir = getSunDirection();
    float special_trick = 1.0 / (dir.y * 0.8 + 0.1); // Increased brightness
    float special_trick2 = 1.0 / (sundir.y * 8.0 + 1.0); // Adjusted for more contrast
    float raysundt = pow(abs(dot(sundir, dir)), 2.0);
    float sundt = pow(max(0.0, dot(sundir, dir)), 6.0); // Reduced for smoother transition
    float mymie = sundt * special_trick * 0.4; // Increased specular intensity
    vec3 suncolor = mix(vec3(1.0, 0.6, 0.3), max(vec3(0.0), vec3(1.0) - vec3(5.5, 13.0, 22.4) / 22.4), special_trick2);
    vec3 bluesky = vec3(5.5, 13.0, 22.4) / 22.4 * suncolor;
    vec3 bluesky2 = max(vec3(0.0), bluesky - vec3(5.5, 13.0, 22.4) * 0.0015 * (special_trick + -4.0 * sundir.y * sundir.y));
    bluesky2 *= special_trick * (0.28 + raysundt * 0.28); // Increased brightness
    
    // Add purple hue to darker areas
    float darkness = 1.0 - smoothstep(0.0, 0.5, dir.y);
    vec3 purpleTint = vec3(0.6, 0.2, 0.8) * darkness * 0.4;
    return (bluesky2 + purpleTint) * (1.0 + 1.2 * pow(1.0 - dir.y, 3.0));
}

// Get sun color for given direction
float getSun(vec3 dir) { 
  return pow(max(0.0, dot(dir, getSunDirection())), 720.0) * 210.0;
}

// Great tonemapping function from my other shader: https://www.shadertoy.com/view/XsGfWV
vec3 aces_tonemap(vec3 color) {  
    color = mix(color, vec3(0.1, 0.05, 0.02), 0.3); // Blend in darker orange
    mat3 m1 = mat3(
        0.59719, 0.07600, 0.02840,
        0.35458, 0.90834, 0.13383,
        0.04823, 0.01566, 0.83777
    );
    mat3 m2 = mat3(
        1.60475, -0.10208, -0.00327,
        -0.53108,  1.10813, -0.07276,
        -0.07367, -0.00605,  1.07602
    );
    vec3 v = m1 * color;  
    vec3 a = v * (v + 0.0245786) - 0.000090537;
    vec3 b = v * (0.983729 * v + 0.4329510) + 0.238081;
    return pow(clamp(m2 * (a / b), 0.0, 1.0), vec3(1.0 / 2.2));  
}

// New function to compute foam intensity based on wave value
float computeFoam(vec2 pos, float depth) {
  float waveValue = getwaves(pos, WAVE_NORMAL_DETAIL);
  // Use smoothstep to create a foam factor when waveValue is in the range [foamThresholdStart, foamThresholdEnd]
  return smoothstep(FOAM_THRESHOLD_START, FOAM_THRESHOLD_END, waveValue);
}

//------------------------------------------------------------------------------
// MAIN RENDERING
//------------------------------------------------------------------------------

//void mainImage( out vec4 fragColor, in vec2 fragCoord )
///////////////////////////////////////////////////////////////////////////////// 
// need to convert this from a void to a function and call it by adding
// a void main(void) { to the end of the shader
// what type of variable will the function return?, it is a color and needs to be a vec4
// change void to vec4 
//void MainImage(out vec4 fragColor, in vec2 fragCoord) 
vec4 mainImage( out vec4 fragColor, in vec2 fragCoord )
{ 
  // Precompute invariant trigonometric functions
  float halfTime = iTime * 0.2;
  float sinHalfTime = sin(halfTime);
  float cosHalfTime = cos(halfTime);

  // get the ray
  vec3 ray = getRay(fragCoord);
  if(ray.y >= 0.0) {
    // if ray.y is positive, render the sky
    vec3 C = computeSunsetAtmosphere(ray, getSunDirection()) + getSun(ray);
    fragColor = vec4(aces_tonemap(C * 2.0),1.0);   
    return fragColor;
  }

  // now ray.y must be negative, water must be hit
  // define water planes
  vec3 waterPlaneHigh = vec3(0.0, 0.0, 0.0);
  vec3 waterPlaneLow = vec3(0.0, -WATER_DEPTH, 0.0);

  // define ray origin, moving around
  vec3 origin = vec3(iTime * 0.2, CAMERA_HEIGHT, 1);

  // calculate intersections and reconstruct positions
  float highPlaneHit = intersectPlane(origin, ray, waterPlaneHigh, vec3(0.0, 1.0, 0.0));
  float lowPlaneHit = intersectPlane(origin, ray, waterPlaneLow, vec3(0.0, 1.0, 0.0));
  vec3 highHitPos = origin + ray * highPlaneHit;
  vec3 lowHitPos = origin + ray * lowPlaneHit;

  // raymatch water and reconstruct the hit pos
  float dist = raymarchwater(origin, highHitPos, lowHitPos, WATER_DEPTH);
  vec3 waterHitPos = origin + ray * dist;

  // calculate normal at the hit position
  vec3 N = normal(waterHitPos.xz, 0.01, WATER_DEPTH);

  // Add foam effect based on wave values
  float foamFactor = computeFoam(waterHitPos.xz, WATER_DEPTH);
  vec3 foamColor = vec3(1.0); // white foam
  // Assuming a base water color; adjust as needed
  vec3 baseWaterColor = vec3(0.04, 0.05, 0.08); // Darker water color
  // Blend foam into base water color based on foam factor
  vec3 waterColor = mix(baseWaterColor, foamColor, foamFactor);

  // Compute depth factor (0.0 for deep water, 1.0 for shallow water)
  float depthFactor = clamp((waterHitPos.y + WATER_DEPTH) / uMaxDepth, 0.0, 1.0);
  // Mix brightness: brighter for shallow water, darker for deep water
  float brightness = mix(0.8, 1.2, depthFactor);
  waterColor *= brightness;

  // smooth the normal with distance to avoid disturbing high frequency noise
  N = mix(N, vec3(0.0, 1.0, 0.0), 0.8 * min(1.0, sqrt(dist*0.01) * 1.1));

  // calculate fresnel coefficient
  float fresnel = (0.04 + (1.0-0.04)*(pow(1.0 - max(0.0, dot(-N, ray)), 5.0)));

  // reflect the ray and make sure it bounces up
  vec3 R = normalize(reflect(ray, N));
  R.y = abs(R.y);
  
  // calculate the reflection and approximate subsurface scattering
  vec3 reflection = computeSunsetAtmosphere(R, getSunDirection()) + getSun(R);
  vec3 scattering = vec3(0.0293, 0.0698, 0.1717) * 0.1 * (0.2 + (waterHitPos.y + WATER_DEPTH) / WATER_DEPTH);

  // return the combined result
  vec3 C = fresnel * reflection + scattering + waterColor;
  fragColor = vec4(aces_tonemap(C * 2.0), 1.0);
/////////////////////////////////////////////////////////////////////////////////
//the function needs to return a value. 
//it needs to be a vec4
//we will return the varable fragColor 
// usual place for fragColor = vec4( color, 1.0 ); bring the } down below 
return fragColor; 
}

///////////////////////////////////////////////////////////////////////////////// 
void main(void) { // this will be run for every pixel of gl_FragCoord.xy
vec4 vTexCoord = gl_TexCoord[0];
vec4 fragColor = vec4(1.0); // initialize variable fragColor as a vec4 
vec4 cc = mainImage(fragColor, gl_FragCoord.xy); // call function mainImage and assign the return vec4 to cc
gl_FragColor = vec4(cc) * gl_Color; // set the pixel to the value of vec4 cc  and..
//gl_FragColor.a = length(gl_FragColor.rgb);
}

// ..uses the values of any Color: or Opacity:
// clauses (and any Animate clauses applied to these properties) 
// appearing in the Sprite, Quad or other node invoking the shader 
// in the .scn file.

